/** @file   tank.h
 * @brief   Declaraction of Tank - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_TANK_H
#define H_WWW_TANK_H

#include "polygonobject.h"
#include "soundsource.h"
#include "MCapturable.h"


namespace WeWantWar {

/** @class  Tank
 * @brief   Represents the tank
 * @author  Tomi Lamminsaari
 *
 * GameObjects can capture the Tank. This is implemented as follows. The
 * GameObject will go to GameObject::HIBERNATING - state when it does not
 * respond any controlling command. When the main update-loop updates this
 * tank we take the controller of the object that has captured us and
 * use it control us.
 */
class Tank : public PolygonObject, public MCapturable
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================
  
  /** Types of the tanks.
   */
  enum Type {
    T_TANK1
  };

  /** Index of the counter that counts the reloading times. */
  static const int RELOADING_COUNTER_INDEX = 0;
  /** Invulnerability counter */
  static const int INVULNERABILITY_COUNTER_INDEX = 1;
  /** Index of the counter that times the no-exit delays */
  static const int NOEXIT_COUNTER_INDEX = 2;


  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   * @param     t                 Type of this tank
   */
	Tank( Type t );


	/** Destructor
   */
	virtual ~Tank();

private:

	/** Copy constructor.
   * @param     rO                Reference to another Tank
   */
  Tank( const Tank& rO );

	/** Assignment operator
   * @param     rO                Reference to another Tank
   * @return    Reference to us.
   */
  Tank& operator = ( const Tank& rO );

public:


  ///
  /// Methods
  /// =======

  /** Updates this object. If this tank is captured by a GameObject we
   * move that object as we move this tank.
   */
  virtual void update();
  
  /** Adds the graphics of this object to the redraw queue.
   * @param     pQueue            Pointer to redraw queue.
   */
  virtual void redraw( RedrawQueue* pQueue );
  
  /** Makes the sound
   * @param     id                ID of the sound.
   */
  virtual void makeSound( GameObject::SoundID id ) const;
  
  /** Destroys this tank.
   */
  virtual void kill();
  
  /** Handles the bullethits
   * @param     pB                Pointer to bullet that hit us
   * @return    <code>true</code> if we accept the bullet hit.
   */
  virtual bool hitByBullet( Bullet* pB );
  
  /** This tank gets captured by the given GameObject.
   * @param     pObj              Pointer to gameobject that captured this
   *                              tank. If null-pointer this tank gets released.
   */
  virtual void capture( GameObject* pObj );

  /** Sets the flag that indicates if this tank can be captured.
   * @param     cap               Set @c true to allow gameobjects to capture
   *                              this tank.
   */
  void setCapture( bool cap );
  
  /** Rotates the cannon angle.
   * @param     a                 Amount of rotation.
   */
  void rotateCannonTower( int a );
  
  
  /** Resurrects us.
   */
  virtual void resurrect();
  
  /** We reimplement this so that we can disable the force-vector
   * @param     rF                The force-vector
   */
  virtual void setForce( const eng2d::Vec2D& rF );
  
  
  

  ///
  /// Getter methods
  /// ==============

  /** Tells if we're reloading our cannon.
   * @return    <code>true</code> if reloading
   */
  virtual bool reloading() const;
  
  /** Returns the type of this object
   * @return    ObjectID::TYPE_TANK
   */
  virtual ObjectID::Type objectType() const;
  
  
  /** Tells if this tank can be captured.
   * @return    <code>true</code> if this object can be captured.
   */
  virtual bool canBeCaptured() const;
  
  /** Returns the pointer to object who has captured us.
   * @return  Pointer to our master or nullpointer if we are not captured by
   *          anyone.
   */
  virtual GameObject* getMaster() const;
  
  /** Returns angle of the cannon tower.
   * @return    Which direction the cannon points
   */
  int cannonAngle() const;
  
  

protected:

  /** Initializes the tank
   */
  void initTank();

  /** This tank has been captured so we use the controller of the object
   * who has captured us.
   */
  void updateCaptured();
  
  /** Shoots
   */
  void shoot();
  
  

  ///
  /// Members
  /// =======

  /** Tells if this can be captured by a GameObject */
  bool m_canCapture;

  /** Pointer to gameobject that has captured this tank */
  GameObject* m_pCapturedBy;

  /** The original cannontower polygon will be here */
  Polygon m_towerPoly;
  
  /** Which direction the cannontower points */
  int   m_cannonAngle;
  
  /** Type of this tank */
  Type  m_type;
  
  /** Speed */
  float m_speed;
  
  /** The reloading delay */
  int m_reloadDelay;
  
  /** Soundsource for moving sound */
  eng2d::SoundSource* m_pMoveSound;

private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: tank.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:18+02  lamminsa
 * Initial revision
 *
 */
 
